/**
 * @fileOverview  The model class Author with property definitions, (class-level) check methods, 
 *                setter methods, and the special methods saveAll and retrieveAll
 * @author Gerd Wagner
 */
import { cloneObject } from "../../lib/util.mjs";
import { NoConstraintViolation, MandatoryValueConstraintViolation,
  RangeConstraintViolation, UniquenessConstraintViolation,
  ReferentialIntegrityConstraintViolation }
  from "../../lib/errorTypes.mjs";

/**
 * The class Author
 * @class
 * @param {object} slots - Object creation slots.
 */
class Author {
  // using a single record parameter with ES6 function parameter destructuring
  constructor ({authorId, name}) {
    // assign properties by invoking implicit setters
    this.authorId = authorId;  // number (integer)
    this.name = name;  // string
    // derived inverse reference property (inverse of Book::authors)
    this._authoredBooks = {};  // initialize as an empty map of Book objects
  }
  get authorId() {
    return this._authorId;
  }
  static checkAuthorId( id) {
    if (!id) {
      return new NoConstraintViolation();  // may be optional as an IdRef
    } else {
      id = parseInt( id);  // convert to integer
      if (isNaN( id) || !Number.isInteger( id) || id < 1) {
        return new RangeConstraintViolation("The author ID must be a positive integer!");
      } else {
        return new NoConstraintViolation();
      }
    }
  }
  static checkAuthorIdAsId( id) {
    var constraintViolation = Author.checkAuthorId(id);
    if ((constraintViolation instanceof NoConstraintViolation)) {
      id = parseInt( id);  // convert to integer
      if (isNaN(id)) {
        return new MandatoryValueConstraintViolation(
            "A positive integer value for the author ID is required!");
      } else if (Author.instances[id]) {
        constraintViolation = new UniquenessConstraintViolation(
            "There is already an author record with this author ID!");
      } else {
        constraintViolation = new NoConstraintViolation();
      }
    }
    return constraintViolation;
  }
  static checkAuthorIdAsIdRef( id) {
    var constraintViolation = Author.checkAuthorId( id);
    if ((constraintViolation instanceof NoConstraintViolation) &&
        id !== undefined) {
      if (!Author.instances[String(id)]) {
        constraintViolation = new ReferentialIntegrityConstraintViolation(
            "There is no author record with this author ID!");
      }
    }
    return constraintViolation;
  }
  set authorId( id) {
    var constraintViolation = Author.checkAuthorIdAsId( id);
    if (constraintViolation instanceof NoConstraintViolation) {
      this._authorId = parseInt( id);
    } else {
      throw constraintViolation;
    }
  }
  get name() {
    return this._name;
  }
  set name( n) {
    /*SIMPLIFIED CODE: no validation with Author.checkName */
    this._name = n;
  }
  get authoredBooks() {
    return this._authoredBooks;
  }
  // Convert object to record with ID references
  toJSON() {  // is invoked by JSON.stringify in Author.saveAll
    var rec = {};
    // loop over all Author properties
    for (const p of Object.keys( this)) {
      // keep underscore-prefixed properties except "_authoredBooks"
      if (p.charAt(0) === "_" && p !== "_authoredBooks") {
        // remove underscore prefix
        rec[p.substr(1)] = this[p];
      }
    }
    return rec;
  }
}
/****************************************************
 *** Class-level ("static") properties ***************
 *****************************************************/
// initially an empty collection (in the form of a map)
Author.instances = {};

/**********************************************************
 ***  Class-level ("static") storage management methods ***
 **********************************************************/
/**
 *  Create a new author record/object
 */
Author.add = function (slots) {
  try {
    const author = new Author( slots);
    Author.instances[author.authorId] = author;
    console.log(`Saved: ${author.name}`);
  } catch (e) {
    console.log(`${e.constructor.name}: ${e.message}`);
  }
};
/**
 *  Update an existing author record
 */
Author.update = function ({authorId, name}) {
  const author = Author.instances[String( authorId)],
        objectBeforeUpdate = cloneObject( author);
  var noConstraintViolated = true, ending = "", updatedProperties = [];
  try {
    if (name && author.name !== name) {
      author.name = name;
      updatedProperties.push("name");
    }
  } catch (e) {
    console.log(`${e.constructor.name}: ${e.message}`);
    noConstraintViolated = false;
    // restore object to its state before updating
    Author.instances[String(authorId)] = objectBeforeUpdate;
  }
  if (noConstraintViolated) {
    if (updatedProperties.length > 0) {
      ending = updatedProperties.length > 1 ? "ies" : "y";
      console.log(`Propert${ending} ${updatedProperties.toString()} modified for author ${name}`);
    } else {
      console.log(`No property value changed for author ${name}!`);

    }
  }
};
/**
 *  Delete an existing author record
 *  Since the book-author association is bidirectional, an author can be directly
 *  deleted from the books' authors, instead of doing a linear search on all books
 *  as required for the case of a unidirectional association.
 */
Author.destroy = function (authorId) {
  const author = Author.instances[authorId];
  // delete all dependent book records
  for (const isbn of Object.keys( author.authoredBooks)) {
    let book = author.authoredBooks[isbn];
    if (book.authors[authorId]) delete book.authors[authorId];
  }
  // delete the author object
  delete Author.instances[authorId];
  console.log(`Author ${author.name} deleted.`);
};
/**
 *  Load all author records and convert them to objects
 */
Author.retrieveAll = function () {
  var authors = {};
  if (!localStorage["authors"]) localStorage["authors"] = "{}";
  try {
    authors = JSON.parse( localStorage["authors"]);
  } catch (e) {
    console.error( "Error when reading from Local Storage\n" + e);
    authors = {};
  }
  for (const key of Object.keys( authors)) {
    try {
      Author.instances[key] = new Author( authors[key]);
    } catch (e) {
      console.log(`${e.constructor.name} while deserializing author ${key}: ${e.message}`);
    }
  }
  console.log(`${Object.keys( authors).length} author records loaded.`);
};
/**
 *  Save all author objects as records
 */
Author.saveAll = function () {
  const keys = Object.keys( Author.instances);
  try {
    localStorage["authors"] = JSON.stringify( Author.instances);
    console.log(`${keys.length} author records saved.`);
  } catch (e) {
    console.error("Error when writing to Local Storage\n" + e);
  }
};

export default Author;
