package pl.model;

import java.util.List;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.faces.validator.ValidatorException;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.Id;
import javax.persistence.Query;
import javax.persistence.Table;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

import pl.model.exception.UniquenessConstraintViolation;

@Entity @Table( name="publishers")
@ViewScoped @ManagedBean( name="publisher")
public class Publisher {
  @Id @NotNull( message="A name is required!")
  private String name;
  @Column( nullable=false)
  @NotNull( message="An address is required!")
  private String address;

  /**
   * Default constructor, required for entity classes
   */
  public Publisher() {}

  /**
   * Constructor
   */
  public Publisher( String name, String address) {
    this.setName( name);
    this.setAddress( address);
  }

  /**
   * Getters and setters
   */
  public String getName() {
    return name;
  }

  public void setName( String name) {
    this.name = name;
  }

  public String getAddress() {
    return address;
  }

  public void setAddress( String address) {
    this.address = address;
  }

  /**
   * Create a human readable serialization.
   */
  public String toString() {
    return "{ name: '" + this.name + "', address:'" + this.address + "'}";
  }
  
  @Override
  public boolean equals( Object obj) {
    if (obj instanceof Publisher) {
      Publisher publisher = (Publisher) obj;
      return ( this.name.equals( publisher.name));
    } else return false;
  }

  /**
   * Check for the name uniqueness constraint by verifying the existence in the
   * database of a publisher entry for the given name value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param name
   *          the name of the publisher to check if exists or not
   * @throws ValidatorException
   * @throws UniquenessConstraintViolation
   */
  public static void checkNameAsId( EntityManager em, String name)
      throws UniquenessConstraintViolation {
    Publisher publisher = Publisher.retrieve( em, name);
    // publisher was found, uniqueness constraint validation failed
    if ( publisher != null) {
      throw new UniquenessConstraintViolation(
          "There is already a publisher record with this name!");
    }
  }

  /**
   * Retrieve all Publisher records from the publishers table.
   * 
   * @param em
   *          reference to the entity manager
   * @return the list of all Publisher records
   */
  @SuppressWarnings( "unchecked")
  public static List<Publisher> retrieveAll( EntityManager em) {
    Query query = em.createQuery( "SELECT p FROM Publisher p", Publisher.class);
    List<Publisher> publishers = query.getResultList();
    System.out.println( "Publisher.retrieveAll: " + publishers.size()
        + " publishers were loaded from DB.");
    return publishers;
  }

  /**
   * Retrieve a publisher record from the publishers table.
   * 
   * @param em
   *          reference to the entity manager
   * @param name
   *          the publisher's name
   * @return the publisher with the given name
   */
  public static Publisher retrieve( EntityManager em, String name) {
    Publisher publisher = em.find( Publisher.class, name);
    if ( publisher != null) {
      System.out.println( "Publisher.retrieve: loaded publisher "
          + publisher);
    }
    return publisher;
  }

  /**
   * Create a Publisher instance.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param name
   *          the name value of the publisher to create
   * @param address
   *          the address value of the publisher to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut, String name,
      String address) throws NotSupportedException, SystemException,
      IllegalStateException, SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException, EntityExistsException {
    ut.begin();
    Publisher publisher = new Publisher( name, address);
    em.persist( publisher);
    ut.commit();
    System.out.println( "Publisher.add: the publisher " + publisher
        + " was saved.");
  }

  /**
   * Update a Publisher instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param name
   *          the name value of the publisher to update
   * @param address
   *          the address value of the publisher to update
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut, String name,
      String address) throws NotSupportedException, SystemException,
      IllegalStateException, SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException {
    ut.begin();
    Publisher publisher = em.find( Publisher.class, name);
    if ( publisher == null) {
      throw new EntityNotFoundException( "The publisher with name = " + name
          + " was not found!");
    }
    if ( address != null && !address.equals( publisher.address)) {
      publisher.setAddress( address);
    }
    ut.commit();
    System.out.println( "Publisher.update: the publisher " + publisher
        + " was updated.");
  }

  /**
   * Delete a Publisher instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param name
   *          the name value of the publisher to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  @SuppressWarnings( "unchecked")
  public static void destroy( EntityManager em, UserTransaction ut, String name)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Publisher publisher = em.find( Publisher.class, name);
    // find all Books which have this publisher
    Query query = em.createQuery(
	    "SELECT b FROM Book b WHERE b.publisher.name = :name");
    query.setParameter( "name", name);
    List<Book> books = query.getResultList();
    // clear these books' publisher reference
    for ( Book b : books) {
      b.setPublisher( null);
    }
    em.remove( publisher);
    ut.commit();
    System.out.println( "Publisher.destroy: the publisher " + publisher
        + " was deleted.");
  }

  /**
   * Clear all entries from the <code>publishers</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void clearData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Query deleteQuery = em.createQuery( "DELETE FROM Publisher");
    deleteQuery.executeUpdate();
    ut.commit();
  }

  /**
   * Create test data (rows) in the <code>publishers</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void createTestData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    Publisher publisher = null;
    // clear existing publishers, so no primary key duplicate conflicts appear
    Publisher.clearData( em, ut);
    ut.begin();
    publisher = new Publisher( "Bantam Books", "New York, USA");
    em.persist( publisher);
    publisher = new Publisher( "Basic Books", "New York, USA");
    em.persist( publisher);
    ut.commit();
  }
}