package pl.model;

import java.util.List;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.faces.validator.ValidatorException;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.Id;
import javax.persistence.Query;
import javax.persistence.Table;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

import pl.model.exception.UniquenessConstraintViolation;

@Entity @Table( name="authors")
@ViewScoped @ManagedBean( name="author")
public class Author {
  @Id @NotNull( message="A person id is required!")
  private Integer personId;
  @Column( nullable=false)
  @NotNull( message="A name is required!")
  private String name;

  /**
   * Default constructor, required for entity classes
   */
  public Author() {}

  /**
   * Constructor
   */
  public Author( Integer personId, String name) {
    this.setPersonId( personId);
    this.setName( name);
  }

  public Integer getPersonId() {
    return personId;
  }

  public void setPersonId( Integer personId) {
    this.personId = personId;
  }

  public String getName() {
    return name;
  }

  public void setName( String name) {
    this.name = name;
  }

  /**
   * Create a human readable serialization.
   */
  public String toString() {
    String result = "{ personId: " + this.personId 
        + ", name:'" + this.name + "'}";
    return result;
  }
  
  @Override
  public boolean equals( Object obj) {
    if (obj instanceof Author) {
      Author author = (Author) obj;
      return ( this.personId.equals( author.personId));
    } else return false;
  }

  /**
   * Check for the personId uniqueness constraint by verifying the existence in
   * the database of an author entry for the given personId value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param personId
   *          the personId of the author to check if exists or not
   * @throws ValidatorException
   * @throws UniquenessConstraintViolation
   */
  public static void checkPersonIdAsId( EntityManager em, Integer personId)
      throws UniquenessConstraintViolation {
    Author author = Author.retrieve( em, personId);
    // author was found, uniqueness constraint validation failed
    if ( author != null) {
      throw new UniquenessConstraintViolation(
          "There is already an author record with this personId!");
    }
  }

  /**
   * Read all the Author entries from the <code>authors</code> database table.
   * 
   * @param em
   *          reference to the entity manager
   * 
   * @return the list of all the Author entries found in the database.
   */
  @SuppressWarnings( "unchecked")
  public static List<Author> retrieveAll( EntityManager em) {
    Query query = em.createQuery( "SELECT p FROM Author p", Author.class);
    List<Author> authors = query.getResultList();
    System.out.println( "Author.retrieveAll: " + authors.size()
        + " authors were loaded from DB.");
    return authors;
  }

  /**
   * Retrieve an author record from the authors table.
   * 
   * @param em
   *          reference to the entity manager
   * 
   * @return the author with the given personId entry found in the database.
   */
  public static Author retrieve( EntityManager em, Integer personId) {
    Author author = em.find( Author.class, personId);
    if ( author != null) {
      System.out.println( "Author.retrieve: loaded author " + author);
    }
    return author;
  }

  /**
   * Create an Author instance.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the author to create
   * @param name
   *          the name value of the author to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut,
      Integer personId, String name)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException, EntityExistsException {
    ut.begin();
    Author author = new Author( personId, name);
    em.persist( author);
    ut.commit();
    System.out.println( "Author.add: the author " + author + " was saved.");
  }

  /**
   * Update an Author instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the author to update
   * @param name
   *          the name value of the author to update
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut,
      Integer personId, String name)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Author author = em.find( Author.class, personId);
    if ( author == null) {
      throw new EntityNotFoundException( "The author with ID = " + personId
          + " was not found!");
    }
    if ( name != null && !name.equals( author.name)) {
      author.setName( name);
    }
    ut.commit();
    System.out
        .println( "Author.update: the author " + author + " was updated.");
  }

  /**
   * Delete an Author instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the author row to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  @SuppressWarnings( "unchecked")
  public static void destroy( EntityManager em, UserTransaction ut,
      Integer personId) throws NotSupportedException, SystemException,
      IllegalStateException, SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException {
    ut.begin();
    Author author = em.find( Author.class, personId);
    // find all books with this author
    Query query = em
        .createQuery( "SELECT DISTINCT b FROM Book b INNER JOIN b.authors a WHERE a.personId = :personId");
    query.setParameter( "personId", personId);
    List<Book> books = query.getResultList();
    // update the corresponding book-to-author relations from the association
    // table (otherwise the author can't be deleted)
    for ( Book b : books) {
      b.getAuthors().remove( author);
    }
    // remove the author entry (table row)
    em.remove( author);
    ut.commit();
    System.out.println( "Author.destroy: the author " + author
        + " was deleted.");
  }

  /**
   * Clear all entries from the <code>authors</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void clearData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Query deleteQuery = em.createQuery( "DELETE FROM Author");
    deleteQuery.executeUpdate();
    ut.commit();
  }

  /**
   * Create test data (rows) in the <code>authors</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void createTestData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    Author author = null;
    // clear existing authors, so no primary key duplicate conflicts appear
    Author.clearData( em, ut);
    ut.begin();
    author = new Author( 1, "Daniel Dennett");
    em.persist( author);
    author = new Author( 2, "Douglas Hofstadter");
    em.persist( author);
    author = new Author( 3, "Immanuel Kant");
    em.persist( author);
  
    ut.commit();
  }
}