package pl.model;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

@Entity
@DiscriminatorValue( value = "BIOGRAPHY")
@ViewScoped
@ManagedBean( name = "biographyBook")
public class Biography extends Book {
  @NotNull( message = "An about value is required")
  private String about;

  /**
   * Default constructor, required for entity classes
   */
  public Biography() {
  }

  /**
   * Constructor
   */
  public Biography( String isbn, String title, Integer year, String about) {
    super( isbn, title, year);
    this.setAbout( about);
  }

  /**
   * Getters and setters
   */
  public String getAbout() {
    return about;
  }

  public void setAbout( String about) {
    this.about = about;
  }

  /**
   * Create a human readable serialisation.
   */
  public String toString() {
    String result = super.toString();
    result = result.substring( 0, result.length() - 2);
    return result + ", about: '" + this.about + "'}";
  }

  /**
   * Retrieve a Biography record from the database.
   * 
   * @param em
   *          reference to the entity manager
   * @param isbn
   *          the book's ISBN
   * @return the biography book with the given ISBN
   */
  public static Biography retrieve( EntityManager em, String isbn) {
    Biography biography = em.find( Biography.class, isbn);
    if ( biography != null) {
      System.out.println( "Biography.retrieve: loaded biography book "
          + biography);
    }
    return biography;
  }

  /**
   * Create a Biography book instance.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to create
   * @param title
   *          the title value of the book to create
   * @param year
   *          the year value of the book to create
   * @param about
   *          the about of the book to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year, String about) throws NotSupportedException,
      SystemException, IllegalStateException, SecurityException,
      HeuristicMixedException, HeuristicRollbackException, RollbackException,
      EntityExistsException {
    ut.begin();
    Biography book = new Biography( isbn, title, year, about);
    em.persist( book);
    ut.commit();
    System.out.println( "Biography.add: the biography book " + book
        + " was saved.");
  }

  /**
   * Update a Biography book instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to update
   * @param title
   *          the title value of the book to update
   * @param year
   *          the year value of the book to create
   * @param about
   *          the about of the book to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year, String about) throws NotSupportedException,
      SystemException, IllegalStateException, SecurityException,
      HeuristicMixedException, HeuristicRollbackException, RollbackException {
    ut.begin();
    Biography book = em.find( Biography.class, isbn);
    if ( book == null) {
      throw new EntityNotFoundException( "The Book with ISBN = " + isbn
          + " was not found!");
    }
    if ( title != null && !title.equals( book.getTitle())) {
      book.setTitle( title);
    }
    if ( year != null && !year.equals( book.getYear())) {
      book.setYear( year);
    }
    if ( about != null && !about.equals( book.getAbout())) {
      book.setAbout( about);
    }
    ut.commit();
    System.out.println( "Biography.update: the biography book " + book
        + " was updated.");
  }

  /**
   * Delete a Biography book instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut, String isbn)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Biography book = em.find( Biography.class, isbn);
    em.remove( book);
    ut.commit();
    System.out.println( "Biography.destroy: the biography book " + book
        + " was deleted.");
  }
}