package pl.ctrl;

import java.util.List;

import javax.annotation.Resource;
import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.SessionScoped;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.validator.ValidatorException;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;

import pl.model.Author;
import pl.model.exception.UniquenessConstraintViolation;

@SessionScoped @ManagedBean( name="authorCtrl")
public class AuthorController {
  @PersistenceContext( unitName="UnidirAssocApp")
  private EntityManager em;
  @Resource private UserTransaction ut;

  /**
   * Get a reference to the entity manager
   * 
   * @return reference to the entity manager
   */
  public EntityManager getEntityManager() {
    return this.em;
  }

  /**
   * Read the list of all authors from the database.
   * 
   * @return an instance of all the author entries found in the database.
   */
  public List<Author> getAuthors() {
    return Author.retrieveAll( em);
  }

  /**
   * Update the reference object by setting its property values to match the one
   * existing in the database for the specific instance, identified by the
   * primary key value.
   * 
   * @param publisher
   *          the reference to the Publisher instance to be "loaded" from
   *          database.
   */
  public void refreshObject( Author author) {
    Author foundAuthor = Author.retrieve( em, author.getPersonId());
    author.setPersonId( foundAuthor.getPersonId());
    author.setName( foundAuthor.getName());
  }

  /**
   * UI specific check for the personId uniqueness constraint. It uses the
   * <code>Author.checkPersonIdsId</code> method to verify the existence in the
   * database of an author entry for the given personId value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param value
   *          the value to be checked - in this case is the author personId to
   *          look for in the database
   * @throws ValidatorException
   */
  public void checkPersonIdAsId( FacesContext context, UIComponent component,
      Object value) throws ValidatorException {
    Integer personId = (Integer) value;
    try {
      Author.checkPersonIdAsId( em, personId);
    } catch ( UniquenessConstraintViolation e) {
      throw new ValidatorException( new FacesMessage(
          FacesMessage.SEVERITY_ERROR, e.getMessage(), e.getMessage()));
    }
  }

  /**
   * Create and persist a new Author instance.
   * 
   * @param personId
   *          the personId of the author to create
   * @param name
   *          the name of the author to create
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String add( Integer personId, String name) {
    try {
      Author.add( em, ut, personId, name);
      // Enforce clearing the form after creating the Author row.
      // Without this, the form will show the latest completed data
      FacesContext facesContext = FacesContext.getCurrentInstance();
      facesContext.getExternalContext().getRequestMap().remove( "author");
    } catch ( EntityExistsException e) {
      try {
        ut.rollback();
      } catch ( Exception e1) {
        e1.printStackTrace();
      }
      e.printStackTrace();
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "create";
  }

  /**
   * Update an Author instance.
   * 
   * @param personId
   *          the personId of the author to update
   * @param name
   *          the name of the author to update (the author will be identified in
   *          the database by using this value)
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String update( Integer personId, String name) {
    try {
      Author.update( em, ut, personId, name);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "update";
  }

  /**
   * Delete an Author entry from database.
   * 
   * @param personId
   *          the id of the author to delete - used to uniquely identify the
   *          author entry.
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String destroy( Integer personId) {
    try {
      Author.destroy( em, ut, personId);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "delete";
  }

  /**
   * Clear/delete all entries from the <code>authors</code> table
   * 
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   * 
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public String clearData() {
    try {
      Author.clearData( em, ut);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "index";
  }

  /**
   * Create test data (rows) in the <code>authors</code> table
   * 
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   * 
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public String createTestData() {
    try {
      Author.createTestData( em, ut);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "index";
  }
}
