package pl.ctrl;

import java.util.List;
import java.util.Set;

import javax.annotation.Resource;
import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.SessionScoped;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.validator.ValidatorException;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.transaction.UserTransaction;

import pl.model.Author;
import pl.model.Book;
import pl.model.Publisher;
import pl.model.exception.UniquenessConstraintViolation;

@SessionScoped @ManagedBean( name="bookCtrl")
public class BookController {
  @PersistenceContext( unitName="BidirAssocApp")
  private EntityManager em;
  @Resource private UserTransaction ut;

  /**
   * Get a reference to the entity manager
   * 
   * @return reference to the entitiy manager
   */
  public EntityManager getEntityManager() {
    return this.em;
  }

  /**
   * Read all books from the database.
   * 
   * @return an instance of all the book entries found in the database.
   */
  public List<Book> getBooks() {
    return Book.retrieveAll( em);
  }

  /**
   * Update the reference object by setting its property values to match the one
   * existing in the database for the specific instance, identified by the
   * primary key value.
   * 
   * @param book
   *          the reference to the Book instance to be "loaded" from database.
   */
  public void refreshObject( Book book) {
    Book foundBook = Book.retrieve( em, book.getIsbn());
    book.setTitle( foundBook.getTitle());
    book.setYear( foundBook.getYear());
    book.setPublisher( foundBook.getPublisher());
    book.setAuthors( foundBook.getAuthors());
  }

  /**
   * UI specific check for the isbn uniqueness constraint. It uses the
   * <code>Book.checkIsbnAsId</code> method to verify the existence in the
   * database of a book entry for the given isbn value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param value
   *          the value to be checked - in this case is the isbn to look for in
   *          the database
   * @throws ValidatorException
   */
  public void checkIsbnAsId( FacesContext context, UIComponent component,
      Object value) throws ValidatorException {
    String isbn = (String) value;
    try {
      Book.checkIsbnAsId( em, isbn);
    } catch ( UniquenessConstraintViolation e) {
      throw new ValidatorException( new FacesMessage(
          FacesMessage.SEVERITY_ERROR, e.getMessage(), e.getMessage()));
    }
  }

  /**
   * Create and persist a new Book instance.
   * 
   * @param isbn
   *          the isbn of the book to create
   * @param title
   *          the title of the book to create
   * @param year
   *          the year of the book to create
   * @param publisher
   *          the book's publisher
   * @param authors
   *          the authors for the book
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String add( String isbn, String title, Integer year,
      Publisher publisher, Set<Author> authors) {
    try {
      Book.add( em, ut, isbn, title, year, publisher, authors);
      // Enforce clearing the form after creating the Book row.
      // Without this, the form will show the latest completed data
      FacesContext facesContext = FacesContext.getCurrentInstance();
      facesContext.getExternalContext().getRequestMap().remove( "book");
    } catch ( EntityExistsException e) {
      try {
        ut.rollback();
      } catch ( Exception e1) {
        e1.printStackTrace();
      }
      e.printStackTrace();
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "create";
  }

  /**
   * Update a Book instance.
   * 
   * @param isbn
   *          the isbn of the book to update (the book will be identified in the
   *          database by using this value)
   * @param title
   *          the new value for the title property
   * @param year
   *          the new value for the year property
   * @param publisher
   *          the book's publisher
   * @param authors
   *          the authors for the book
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String update( String isbn, String title, Integer year,
      Publisher publisher, Set<Author> authors) {
    try {
      Book.update( em, ut, isbn, title, year, publisher, authors);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "update";
  }

  /**
   * Delete a Book entry from database.
   * 
   * @param isbn
   *          the isbn of the book to delete - used to uniquely identify the
   *          book entry.
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String destroy( String isbn) {
    try {
      Book.destroy( em, ut, isbn);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "delete";
  }
}
